﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace StudyScores
{
    /// <summary>
    /// Stores scaling data for a subject and provides raw to scaled score conversion methods.
    /// </summary>
    class Scaling
    {

        /// <summary>
        /// Holds a list of raw scores (generally 40-50).
        /// </summary>
        public int[] RawScores { get; set; }

        /// <summary>
        /// Holds a list of scaled scores corresponding to <see cref="RawScores"/>.
        /// </summary>
        public double[] ScaledScores { get; set; }

        /// <summary>
        /// Returns <see cref="ScaledScores"/> as an integer array, with numbers rounded.
        /// </summary>
        /// <returns>An integer array containing the list of scaled scores.</returns>
        public int[] GetIntegerScaledScores()
        {
            int[] result = new int[ScaledScores.Length];
            for (int i = 0; i < result.Length; i++)
                result[i] = (int)Math.Round(ScaledScores[i]);
            return result;
        }

        /// <summary>
        /// Retrieves the corresponding scaled score for the specified raw score.
        /// </summary>
        /// <param name="rawscore">The raw score whose scaled equivalent should be returned.</param>
        /// <returns>A scaled score corresponding to the given raw score.</returns>
        public double GetScaledScore(int rawscore)
        {
            if (rawscore < Program.LowestRawScore || rawscore > Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentOutOfRangeException(String.Format("Raw score must be between {0} and {1}. Input: {2}.", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1, rawscore));
            return ScaledScores[rawscore - Program.LowestRawScore];
        }



        /// <summary>
        /// Initialises the Scaling object. Takes two arrays containing raw and scaled scores correspondences.
        /// Outstanding scaled scores will be calculated with linear interpolation.
        /// At minimum, <paramref name="rawscores"/> must contain entries for the minimum and maximum raw scores defined for the program.
        /// </summary>
        /// <param name="rawscores">An array of raw scores. Must contain at least the minimum and maximum raw scores.</param>
        /// <param name="scaledscores">An array of scaled scores. Length must correspond exactly to that of <paramref name="rawscores"/>.</param>
        public Scaling(int[] rawscores, double[] scaledscores)
        {
            RawScores = new int[Program.ScoreLevelCount];
            ScaledScores = new double[Program.ScoreLevelCount];

            if (rawscores[0] != Program.LowestRawScore || rawscores[rawscores.Length - 1] != Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentException(String.Format("Scaling data must be provided for at least the minimum and maximum raw scores ({0} and {1}).", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1));

            
            // set data, interpolating as necessary

            // indicates at what point in the array we should write numbers
            int scoreindex = 1;

            for (int i = 1; i < rawscores.Length; i++)
            {

                RawScores[scoreindex - 1] = rawscores[i - 1];
                ScaledScores[scoreindex - 1] = scaledscores[i - 1];

                int steps = rawscores[i] - rawscores[i - 1]; // generally 5
                double scaleddiff = scaledscores[i] - scaledscores[i - 1];
                double scaledstep = scaleddiff / steps;

                for (int j = 0; j < steps; j++)
                {
                    RawScores[j + scoreindex] = RawScores[scoreindex - 1] + (j + 1);
                    ScaledScores[j + scoreindex] = ScaledScores[scoreindex - 1] + ((j + 1) * scaledstep);
                }

                scoreindex += steps;

            }


        }

    }
}
