﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace StudyScores
{
    class School
    {

        /// <summary>
        /// The school's name.
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// The school's name with characters that would be invalid in a Windows filename removed.
        /// </summary>
        public string SafeName { get { return Name.Replace(":", " -").Replace("/", "-"); } }

        /// <summary>
        /// The number of times each score between 40 and 50 was achieved at the school.
        /// </summary>
        public int[] ScoreCounts { get; set; }

        /// <summary>
        /// List of students who attended this school.
        /// </summary>
        public Dictionary<string, Student> Students { get; set; }

        /// <summary>
        /// Total number of each score scaled. Access score by index (0 to 55 in general).
        /// </summary>
        public int[] ScaledScoreCounts;

        /// <summary>
        /// Sum of the school's raw scores.
        /// </summary>
        public int ScoreSum { get; set; }

        /// <summary>
        /// Sum of the school's scaled scores.
        /// </summary>
        public double ScaledScoreSum { get; set; }



        public School()
        {
            ScoreCounts = new int[Program.ScoreLevelCount];
            ScaledScoreCounts = new int[Program.MaximumScaledScoreSpecial + 1];
            Students = new Dictionary<string, Student>(5000);
        }

        public override string ToString()
        {
            return Name;
        }


        public int GetScoreCount(int rawscore)
        {
            if (rawscore < Program.LowestRawScore || rawscore > Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentOutOfRangeException(String.Format("Score must be between {0} and {1}. Input: {2}.", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1, rawscore));
            return ScoreCounts[rawscore - Program.LowestRawScore];
        }

        /// <summary>
        /// Adds the specified student (if not already present) and score.
        /// </summary>
        /// <param name="student">Student to be added (or ignored if already present).</param>
        /// <param name="score">Score to be added.</param>
        public void AddStudentAndScore(Student student, int score)
        {
            if (score < Program.LowestRawScore || score > Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentOutOfRangeException(String.Format("Score must be between {0} and {1}. Input: {2}.", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1, score));
            if (!Students.ContainsKey(student.Name))
                Students.Add(student.Name, student);
            ScoreCounts[score - Program.LowestRawScore]++;
        }

        public void AddScaledScore(double scaledscore)
        {
            ScaledScoreSum += scaledscore;
            ScaledScoreCounts[(int)Math.Ceiling(scaledscore)]++;
        }



        /// <summary>
        /// Returns a string containing the school name, total number of scores and counts for each raw score.
        /// </summary>
        /// <returns>String containing school name, total number of scores and individual raw sscore counts.</returns>
        public string RawScoreCountsToString()
        {
            StringBuilder result = new StringBuilder();

            // write name and score count
            result.AppendLine(String.Format("{0} ({1} total)", Name, ScoreCounts.Sum()));

            result.Append("\t");

            // number of leading spaces to pad numbers with
            int padlength = 3;

            // write list of score counts
            for (int i = ScoreCounts.Length - 1; i >= 0; i--)
            {
                result.AppendFormat("{0}: {1}; ", i + Program.LowestRawScore, ScoreCounts[i].ToString().PadLeft(padlength, ' '));
            }

            return result.ToString().TrimEnd();
        }



        /// <summary>
        /// Returns a string containing the school name, total number of scores and counts for each raw score.
        /// </summary>
        /// <returns>String containing school name, total number of scores and individual raw sscore counts.</returns>
        public string ScaledScoreCountsToString()
        {

            StringBuilder result = new StringBuilder();

            // write name and score count
            result.AppendLine(String.Format("{0} ({1} total)", Name, ScaledScoreCounts.Sum()));

            result.Append("\t");

            // number of leading spaces to pad numbers with
            int padlength = 3;

            // write list of score counts
            for (int i = ScaledScoreCounts.Length - 1; i >= Program.LowestSeenScaledScore; i--)
            {
                result.AppendFormat("{0}: {1}; ", i, ScaledScoreCounts[i].ToString().PadLeft(padlength, ' '));
            }

            return result.ToString().TrimEnd();

        }



    }
}
