﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace StudyScores
{
    /// <summary>
    /// Represents a subject. Contains references to all students with scores for the subject, and the total number of scores recorded.
    /// </summary>
    class Subject : IComparable
    {

        /// <summary>
        /// The subject's name.
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// List of students linked to this subject.
        /// </summary>
        public List<Student> Students { get; set; }

        /// <summary>
        /// The number of occurrences of each score of 40 or above.
        /// </summary>
        public int[] ScoreCounts { get; set; }

        /// <summary>
        /// Scaling data associated with this subject.
        /// </summary>
        public Scaling ScalingData { get; set; }

        public bool HasScalingData
        {
            get
            {
                return (ScalingData != null);
            }
        }

        /// <summary>
        /// Total number of each score scaled. Access score by index (0 to 55 in general).
        /// </summary>
        static int[] ScaledScoreCounts;

        public Subject()
        {
            // initialise student lists
            Students = new List<Student>(5000);
            ScoreCounts = new int[Program.ScoreLevelCount];
            ScaledScoreCounts = new int[Program.MaximumScaledScoreSpecial + 1];
        }


        public int CompareTo(object obj)
        {
            Subject subject = obj as Subject;
            if (subject != null)
            {
                // put VCE VET last, after VisCom
                if (subject.Name.StartsWith("VCE VET") && !Name.StartsWith("VCE VET"))
                    return -1;
                else if (!subject.Name.StartsWith("VCE VET") && Name.StartsWith("VCE VET"))
                    return 1;
                return Name.CompareTo(subject.Name);
            }
            else
                throw new ArgumentException("Must compare to another Subject.");
        }


        /// <summary>
        /// Adds the specified student (if not already present) and score.
        /// </summary>
        /// <param name="student">Student to be added (or ignored if already present).</param>
        /// <param name="score">Score to be added.</param>
        public void AddStudentAndScore(Student student, int score)
        {
            if (score < Program.LowestRawScore || score > Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentOutOfRangeException(String.Format("Score must be between {0} and {1}. Input: {2}.", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1, score));
            ScoreCounts[score - Program.LowestRawScore]++;
            ScaledScoreCounts[GetScaledScoreAsInt(score)]++;
            Students.Add(student);
        }

        public override string ToString()
        {
            return Name;
        }



        /// <summary>
        /// Returns the scaled equivalent of the given raw score. If no scaling data is present, returns the raw score.
        /// </summary>
        /// <param name="rawscore">The raw score to scale.</param>
        /// <returns>Scaled score corresponding to provided raw score.</returns>
        public double GetScaledScore(int rawscore)
        {
            if (!HasScalingData)
                return rawscore;
            return ScalingData.GetScaledScore(rawscore);
        }

        /// <summary>
        /// Returns the scaled equivalent of the given raw score as an integer. If no scaling data is present, returns the raw score.
        /// </summary>
        /// <param name="rawscore">The raw score to scale.</param>
        /// <returns>Scaled score corresponding to provided raw score.</returns>
        public int GetScaledScoreAsInt(int rawscore)
        {
            if (ScalingData == null)
                return rawscore;
            return (int)Math.Ceiling(ScalingData.GetScaledScore(rawscore));
        }

        /// <summary>
        /// Returns true if the raw score scales above 50 (even if the scaled score rounds down to 50).
        /// </summary>
        /// <param name="rawscore">The raw score to check.</param>
        /// <returns>True if the raw score scales to more than 50, false if not.</returns>
        public bool ScalesAboveMax(int rawscore)
        {
            if (!HasScalingData)
                return false;
            return (ScalingData.GetScaledScore(rawscore) > Program.MaximumScoreStandard);
        }

        /// <summary>
        /// Generates a string representing this subject. Includes a list of all students linked to the subject.
        /// </summary>
        /// <returns>String listing students and score totals.</returns>
        public string StudentListToString()
        {
            StringBuilder result = new StringBuilder();

            result.AppendLine(String.Format("{0} ({1} total)", Name, Students.Count));

            // write scaling data if present
            if (Program.ScalingEnabled) {
                result.AppendLine();
                result.AppendLine(Program.GetScalingLine(this));
            }

            // list students at each score level
            List<Student>[] studentlists = new List<Student>[Program.ScoreLevelCount];
            for (int j = 0; j < studentlists.Length; j++)
                studentlists[j] = new List<Student>(500);

            foreach (Student student in Students)
                studentlists[(int)student.GetScore(this) - Program.LowestRawScore].Add(student);

            // loop through each score level
            for (int j = studentlists.Length - 1; j >= 0; j--)
            {
                // only include scores for which there are students
                if (studentlists[j].Count > 0)
                {

                    result.AppendLine();
                    result.AppendLine(String.Format("{0} ({1} total):", j + Program.LowestRawScore, studentlists[j].Count));

                    // sort student names alphabetically
                    studentlists[j].Sort();

                    foreach (Student student in studentlists[j])
                        if (Program.SchoolDataEnabled)
                            result.AppendLine(String.Format("\t{0} ({1})", student.Name, student.School.Name));
                        else
                            result.AppendLine(String.Format("\t{0}", student.Name));

                }
            }

            return result.ToString().TrimEnd();
        }

    }
}
