﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Text.RegularExpressions;
using System.Diagnostics;

namespace StudyScores
{
    class Program
    {

        #region Constants

        /// <summary>
        /// The lowest score present in the data.
        /// </summary>
        public const int LowestRawScore = 40;

        /// <summary>
        /// The number of different scores. Set to 11 for 40-50 (11 different scores).
        /// </summary>
        public const int ScoreLevelCount = 11;

        /// <summary>
        /// The root data directory.
        /// </summary>
        const string DataDirectory = @"D:\Programming\Study Scores\Data";

        /// <summary>
        /// The root output directory.
        /// </summary>
        const string OutputDirectory = @"D:\Programming\Study Scores\Output";

        /// <summary>
        /// Minimum possible subject score.
        /// </summary>
        public const int MinimumScore = 0;

        /// <summary>
        /// Maximum possible raw subject score.
        /// </summary>
        public const int MaximumScoreStandard = 50;

        /// <summary>
        /// Maximum possible scaled subject score (excluding special subjects).
        /// </summary>
        public const int MaximumScaledScoreStandard = 50;

        /// <summary>
        /// Maximum possible scaled subject score (special subjects).
        /// </summary>
        public const int MaximumScaledScoreSpecial = 55;

        #endregion

        #region Enums

        /// <summary>
        /// List sorting methods.
        /// </summary>
        public enum SortingMethods
        {
            /// <summary>
            /// Sorts according to the highest scores. Deadlock is broken by moving to the next-highest score.
            /// </summary>
            HighestScores,
            /// <summary>
            /// Sorts according to the sum of scores. Two scores outrank a single score.
            /// </summary>
            TotalScores
        }

        /// <summary>
        /// Sorting methods with regards to scaling of scores.
        /// </summary>
        public enum ScalingMethods
        {
            /// <summary>
            /// Sort by raw scores.
            /// </summary>
            RawScores,
            /// <summary>
            /// Sort by scaled scores.
            /// </summary>
            ScaledScores
        }

        /// <summary>
        /// Options for including school names in listings.
        /// </summary>
        public enum SchoolNameOptions
        {
            /// <summary>
            /// Include school name in listing.
            /// </summary>
            IncludeName,
            /// <summary>
            /// Don't include school name in listing.
            /// </summary>
            NoName
        }
        
        #endregion


        /// <summary>
        /// List of subjects.
        /// </summary>
        static List<Subject> Subjects;

        /// <summary>
        /// List of students.
        /// </summary>
        static Dictionary<string, Student> Students;

        /// <summary>
        /// True if school data is present, false if not.
        /// </summary>
        public static bool SchoolDataEnabled;

        /// <summary>
        /// List of schools.
        /// </summary>
        static Dictionary<string, School> Schools;

        /// <summary>
        /// List of schools associated with each subject.
        /// </summary>
        static Dictionary<Subject, Dictionary<string, School>> SubjectSchools;

        /// <summary>
        /// Number of records read.
        /// </summary>
        static int RecordCount;

        /// <summary>
        /// Total number of each score.
        /// </summary>
        static int[] ScoreCounts;

        /// <summary>
        /// True if scaling data is available, false otherwise.
        /// </summary>
        public static bool ScalingEnabled;

        /// <summary>
        /// Holds scaling data, if <see cref="ScalingEnabled"/> is set to true.
        /// </summary>
        static Dictionary<string, Scaling> ScalingData;

        /// <summary>
        /// Total number of each score scaled. Access score by index (0 to 55 in general).
        /// </summary>
        static int[] ScaledScoreCounts;

        /// <summary>
        /// Total number of scores that scale to more than <see cref="MaximumScoreStandard"/>.
        /// </summary>
        static int ScoresAboveMax;

        /// <summary>
        /// Total number of scores scaled down.
        /// </summary>
        static int ScoresScaledDown;

        /// <summary>
        /// Total number of scores scaled up.
        /// </summary>
        static int ScoresScaledUp;

        /// <summary>
        /// Total number of scores scaled without movement.
        /// </summary>
        static int ScoresScaledSideways;

        /// <summary>
        /// Total number of scores not scaled (due to lack of scaling data).
        /// </summary>
        static int ScoresNotScaled;

        /// <summary>
        /// Lowest seen scaled score.
        /// </summary>
        public static int? LowestSeenScaledScore;



        static void Main(string[] args)
        {

            ParseData();

            //Console.ReadLine();

        }



        static void ParseData()
        {

            // ensure directory exists
            if (!Directory.Exists(DataDirectory))
                throw new DirectoryNotFoundException(String.Format("Specified data directory {0} could not be found.", DataDirectory));

            // make a list of the directories
            string[] yeardirectories = Directory.GetDirectories(DataDirectory, "????");

            // read data for each directory
            foreach (string yeardirectory in yeardirectories)
            {

                // only read 4-digit named directories
                if (Regex.IsMatch((new DirectoryInfo(yeardirectory)).Name, "[12][0-9]{3}"))
                {
                    ParseYear(yeardirectory);
                }

            }

        }


        static void ParseYear(string directory)
        {

            Stopwatch timer = new Stopwatch();

            timer.Start();

            Console.Write("Reading directory {0} ... ", directory); 

            //try
            //{
                ReadYear(directory);
            //}
            //catch (Exception ex)
            //{
            //  Console.WriteLine("Failed to read subject data. Details: {0}", ex.Message);
            //  Console.ReadLine();
            //  Environment.Exit(0);
            //}

            timer.Stop();

            Console.WriteLine("done in {0}ms", timer.ElapsedMilliseconds);

            if (Subjects.Count > 0)
            {

                Console.WriteLine("Read {0} scores, {1} students and {2} schools.", ScoreCounts.Sum(), Students.Count, (Schools != null ? Schools.Count : 0));

                Console.Write("Writing output ... ");

                timer.Reset();

                timer.Start();

                //try
                //{
                WriteYear((new DirectoryInfo(directory)).Name);
                //}
                //catch (Exception ex)
                //{
                //    Console.WriteLine("Failed to write subject data. Details: {0}", ex.Message);
                //    Console.ReadLine();
                //    Environment.Exit(0);
                //}

                timer.Stop();

                Console.WriteLine("done in {0}ms", timer.ElapsedMilliseconds);

            }
            else
                Console.WriteLine("Warning: No subject data found in directory {0}", directory);

            Console.WriteLine();

        }



        /// <summary>
        /// Reads data for the specified year's directory.
        /// </summary>
        /// <param name="directory">Directory containing subject data.</param>
        static void ReadYear(string directory)
        {

            // initialise student hashtable
            Students = new Dictionary<string, Student>(25000);

            // reset record count
            RecordCount = 0;

            // reset score counts
            ScoreCounts = new int[ScoreLevelCount];

            // reset scaling data
            string scalingfilepath = Path.Combine(directory, "! Scaling.txt");
            ScalingEnabled = File.Exists(scalingfilepath);
            if (ScalingEnabled)
            {
                LoadScalingData(scalingfilepath);
                ScaledScoreCounts = new int[MaximumScaledScoreSpecial + 1];
                ScoresAboveMax = 0;

                ScoresScaledUp = 0;
                ScoresScaledDown = 0;
                ScoresScaledSideways = 0;
                ScoresNotScaled = 0;

                LowestSeenScaledScore = null;
            }



            // check if reading from CSV
            if (File.Exists(Path.Combine(directory, "Subjects.csv")))
            {
                ReadFromCSV(Path.Combine(directory, "Subjects.csv"));
            }
            else
            {
                ReadFromTextFiles(directory);
            }

        }



        /// <summary>
        /// Reads data from per-subject text files in the specified directory.
        /// </summary>
        /// <param name="directory"></param>
        static void ReadFromTextFiles(string directory)
        {

            // make a list of subject data files
            string[] subjectfiles = Directory.GetFiles(directory, "*.txt");

            // initialise subject list
            Subjects = new List<Subject>(subjectfiles.Length);

            // initialise school hashtable
            Schools = new Dictionary<string, School>(1000);
            SchoolDataEnabled = true;
            SubjectSchools = new Dictionary<Subject, Dictionary<string, School>>(subjectfiles.Length * 2);

            for (int i = 0; i < subjectfiles.Length; i++)
            {

                // retrieve subject name
                string subjectname = Path.GetFileNameWithoutExtension(subjectfiles[i]);

                // ignore any files starting with '!'
                if (!subjectname.StartsWith("!"))
                {

                    // create subject object
                    Subject subject = new Subject() { Name = subjectname };
                    Subjects.Add(subject);

                    // create subject school object
                    SubjectSchools.Add(subject, new Dictionary<string, School>(1000));


                    // set scaling data if it exists
                    if (ScalingEnabled)
                    {
                        if (ScalingData.ContainsKey(subjectname))
                            subject.ScalingData = ScalingData[subjectname];
                    }



                    // the latest score we've seen; null if we've seen none
                    int? currentscore = null;


                    // read all data
                    string[] subjectlines = File.ReadAllLines(subjectfiles[i]);


                    foreach (string line in subjectlines)
                    {

                        string trimmedline = line.Trim();

                        int newscore;
                        if (int.TryParse(trimmedline, out newscore))
                        {
                            // current line contains a score
                            currentscore = newscore;
                        }

                        else if (trimmedline != "" && currentscore != null)
                        {
                            // current line contains student data
                            // but ignore it if it's empty or if we haven't yet seen a score line

                            // limit to 3 substrings, since the schoolname will occasionally have a comma in it
                            string[] splitline = trimmedline.Split(new char[] { ',' }, 3);

                            if (splitline.Length != 3)
                                throw new ArgumentException(String.Format("Malformed line detected. Each data line must define a Surname, First name and School name, separated by commas. Input: {0} ({1})", trimmedline, Path.Combine(directory, subjectfiles[i])));

                            // retrieve student name
                            string studentname = splitline[0] + ',' + splitline[1];

                            // retrieve school name
                            string schoolname = splitline[2].Substring(1, splitline[2].Length - 2);

                            ReadLine(subject, studentname, schoolname, (int)currentscore, SchoolNameOptions.IncludeName);

                        }


                    }

                }

            }

        }



        /// <summary>
        /// Reads data from the specified CSV file.
        /// </summary>
        /// <param name="filename">Path to CSV file containing score data.</param>
        static void ReadFromCSV(string filename)
        {

            // initialise subject list
            // hard coded to 150
            Subjects = new List<Subject>(150);

            string[] csvlines = File.ReadAllLines(filename);

            // initialise school hashtable
            Schools = new Dictionary<string, School>(1000);
            SchoolDataEnabled = true;
            SubjectSchools = new Dictionary<Subject, Dictionary<string, School>>(Subjects.Capacity * 2);

            // read each line of the CSV
            foreach (string line in csvlines)
            {

                string[] linesplit = line.Split(new char[] { ',' }, 6);

                if (linesplit.Length != 6)
                    throw new ArgumentException(String.Format("Malformed line detected. Each CSV line contain a Subject Name, Score, Surname, First Name, School Name and School Location, separated by commas. Input: {0}", line));

                string subjectname = linesplit[0];
                string scorestring = linesplit[1];
                string surname = linesplit[2];
                string firstname = linesplit[3];
                string schoolname = linesplit[4] + ", " + linesplit[5]; // concatenate school name and location
                
                string studentname = surname + ", " + firstname;

                int score;
                if (!int.TryParse(scorestring, out score))
                    throw new ArgumentException(String.Format("Error parsing score as integer. Input: {0}", scorestring));

                Subject subject;

                // create subject object if not present
                if (Subjects.Count == 0 || Subjects[Subjects.Count - 1].Name != subjectname)
                {
                    // create subject object
                    subject = new Subject() { Name = subjectname };
                    Subjects.Add(subject);

                    // create subject school object
                    SubjectSchools.Add(subject, new Dictionary<string, School>(1000));

                    // set scaling data if it exists
                    if (ScalingEnabled)
                    {
                        if (ScalingData.ContainsKey(subjectname))
                            subject.ScalingData = ScalingData[subjectname];
                    }
                }
                else
                {
                    subject = Subjects[Subjects.Count - 1];
                }

                ReadLine(subject, studentname, schoolname, score, SchoolNameOptions.IncludeName);

            }

        }



        /// <summary>
        /// Takes a set of values from a single data line and adds the appropriate records.
        /// </summary>
        /// <param name="subject">Subject object to use in records.</param>
        /// <param name="studentname">Name of student to add to records.</param>
        /// <param name="schoolname">Name of school to add to records.</param>
        /// <param name="score">Raw score to add to records.</param>
        /// <param name="schoolnameoption">Specifies whether the school name is included or not.</param>
        static void ReadLine(Subject subject, string studentname, string schoolname, int score, SchoolNameOptions schoolnameoption)
        {

            // often a student name appears more than once in the data
            // we must thus identify the student by both their name and school
            string studentkey;

            // but in 2010 we don't have the school data
            if (schoolnameoption == SchoolNameOptions.IncludeName)
                studentkey = studentname + schoolname;
            else
                studentkey = studentname;

            School school = null; // set to null to avoid unassigned variable error
            School subjectschool = null;
            Student student;

            if (schoolnameoption == SchoolNameOptions.IncludeName)
            {
                // create school object if it doesn't already exist
                if (!Schools.ContainsKey(schoolname))
                {
                    school = new School() { Name = schoolname };
                    Schools[schoolname] = school;
                }
                else
                {
                    school = Schools[schoolname];
                }

                // create subject school object if it doesn't already exist
                if (!SubjectSchools[subject].ContainsKey(schoolname))
                {
                    subjectschool = new School() { Name = schoolname };
                    SubjectSchools[subject][schoolname] = subjectschool;
                }
                else
                {
                    subjectschool = SubjectSchools[subject][schoolname];
                }
            }

            // create student object if it doesn't already exist
            if (!Students.ContainsKey(studentkey))
            {
                if (schoolnameoption == SchoolNameOptions.IncludeName)
                    student = new Student() { Name = studentname, School = school };
                else
                    student = new Student() { Name = studentname };
                Students[studentkey] = student;
            }
            else
            {
                student = Students[studentkey];
            }



            // add score to student and school and subject

            student.AddScore(subject, score);

            if (schoolnameoption == SchoolNameOptions.IncludeName)
            {
                school.AddStudentAndScore(student, score);
                school.ScoreSum += score;
                if (ScalingEnabled)
                    school.AddScaledScore(subject.GetScaledScore(score));

                subjectschool.AddStudentAndScore(student, score);
                subjectschool.ScoreSum += score;
                if (ScalingEnabled)
                    subjectschool.AddScaledScore(subject.GetScaledScore(score));
            }

            subject.AddStudentAndScore(student, score);




            RecordCount++;

            ScoreCounts[score - LowestRawScore]++;


            if (ScalingEnabled)
            {

                if (LowestSeenScaledScore == null || LowestSeenScaledScore > Math.Ceiling(subject.GetScaledScore(score)))
                    LowestSeenScaledScore = (int)Math.Ceiling(subject.GetScaledScore(score));

                if (subject.ScalesAboveMax(score)) ScoresAboveMax++;

                if (subject.ScalingData != null)
                {
                    double scaledscore = subject.GetScaledScore(score);
                    if (score > scaledscore)
                        ScoresScaledDown++;
                    else if (score < scaledscore)
                        ScoresScaledUp++;
                    else
                        ScoresScaledSideways++;
                }
                else
                    ScoresNotScaled++;

                ScaledScoreCounts[subject.GetScaledScoreAsInt(score)]++;

            }

        }



        /// <summary>
        /// Loads scaling data from the specified scaling file.
        /// </summary>
        /// <param name="scalingfilepath">Location of text file containing scaling data.</param>
        static void LoadScalingData(string scalingfilepath)
        {

            string[] scalinglines = File.ReadAllLines(scalingfilepath);

            ScalingData = new Dictionary<string, Scaling>(scalinglines.Length * 2);

            // read raw scores
            string[] rawscorestrings = scalinglines[0].Split(',');
            int[] rawscores = new int[rawscorestrings.Length];

            for (int i = 0; i <rawscorestrings.Length; i++)
            {
                int score;
                if (!int.TryParse(rawscorestrings[i].Trim(), out score))
                    throw new FormatException(String.Format("The first line of the scaling file must contain integers separated by columns. Failed to parse {0} as an integer. Input was {1}.", rawscorestrings[i].Trim(), scalinglines[0]));
                rawscores[i] = score;
            }


            for (int i = 1; i < scalinglines.Length; i++)
            {

                string[] namescoresplit = scalinglines[i].Split(':');

                if (namescoresplit.Length != 2)
                    throw new FormatException(String.Format("Lines must contain the subject name followed by a colon, followed by the set of scaled scores. Input was {0}.", scalinglines[i]));

                string name = namescoresplit[0].Trim();

                string[] scoresplit = namescoresplit[1].Trim().Split(',');

                double[] scaledscores = new double[scoresplit.Length];

                for (int j = 0; j < scoresplit.Length; j++)
                    if (!double.TryParse(scoresplit[j].Trim(), out scaledscores[j]))
                        throw new FormatException(String.Format("Lines must contain the subject name followed by a colon, followed by the set of scaled scores. Failed to parse {0} as an integer. Input was {1}.", scoresplit[j], scalinglines[i]));

                Scaling scaling = new Scaling(rawscores, scaledscores);

                ScalingData[name] = scaling;

            }

        }



        /// <summary>
        /// Writes lists for the specified year.
        /// </summary>
        /// <param name="directory">The name of the directory</param>
        static void WriteYear(string directory)
        {

            // get output path
            string outputdir = Path.Combine(OutputDirectory, directory);

            // create folder if it doesn't already exist
            Directory.CreateDirectory(outputdir);

            

            // 1. Write Statistics

            string statisticspath = Path.Combine(outputdir, "Statistics.txt");
            File.WriteAllText(statisticspath, StatisticsToString());



            // 2. Write Scaling Data (if present)

            if (ScalingEnabled)
            {
                string scalingpath = Path.Combine(outputdir, "Scaling Table.txt");
                File.WriteAllText(scalingpath, ScalingDataToString());
            }



            // 3. Write Student Lists

            Student[] students = Students.Values.ToArray();

            SchoolNameOptions schoolnameoption = (SchoolDataEnabled ? SchoolNameOptions.IncludeName : SchoolNameOptions.NoName);

            string studenthighscorespath = Path.Combine(outputdir, "All Students, Sorted by Highest Scores (Raw).txt");
            File.WriteAllText(studenthighscorespath, StudentsToString(students, SortingMethods.HighestScores, ScalingMethods.RawScores, schoolnameoption));

            string studenttotalscorespath = Path.Combine(outputdir, "All Students, Sorted by Total Scores (Raw).txt");
            File.WriteAllText(studenttotalscorespath, StudentsToString(students, SortingMethods.TotalScores, ScalingMethods.RawScores, schoolnameoption));

            if (ScalingEnabled)
            {
                string studenthighscoresscaledpath = Path.Combine(outputdir, "All Students, Sorted by Highest Scores (Scaled).txt");
                File.WriteAllText(studenthighscoresscaledpath, StudentsToString(students, SortingMethods.HighestScores, ScalingMethods.ScaledScores, schoolnameoption));

                string studenttotalscoresscaledpath = Path.Combine(outputdir, "All Students, Sorted by Total Scores (Scaled).txt");
                File.WriteAllText(studenttotalscoresscaledpath, StudentsToString(students, SortingMethods.TotalScores, ScalingMethods.ScaledScores, schoolnameoption));
            }



            // 4. Write School Score Counts

            if (SchoolDataEnabled)
            {

                School[] schools = Schools.Values.ToArray();

                string schoolscorecountshighscorespath = Path.Combine(outputdir, "Score Counts, Grouped by School, Sorted by Highest Scores (Raw).txt");
                File.WriteAllText(schoolscorecountshighscorespath, SchoolScoreCountsToString(schools, SortingMethods.HighestScores, ScalingMethods.RawScores));

                string schoolscorecountstotalscorespath = Path.Combine(outputdir, "Score Counts, Grouped by School, Sorted by Total Scores (Raw).txt");
                File.WriteAllText(schoolscorecountstotalscorespath, SchoolScoreCountsToString(schools, SortingMethods.TotalScores, ScalingMethods.RawScores));

                if (ScalingEnabled)
                {
                    string schoolscorecountshighscoresscaledpath = Path.Combine(outputdir, "Score Counts, Grouped by School, Sorted by Highest Scores (Scaled).txt");
                    File.WriteAllText(schoolscorecountshighscoresscaledpath, SchoolScoreCountsToString(schools, SortingMethods.HighestScores, ScalingMethods.ScaledScores));

                    string schoolscorecountstotalscoresscaledpath = Path.Combine(outputdir, "Score Counts, Grouped by School, Sorted by Total Scores (Scaled).txt");
                    File.WriteAllText(schoolscorecountstotalscoresscaledpath, SchoolScoreCountsToString(schools, SortingMethods.TotalScores, ScalingMethods.ScaledScores));
                }

            }



            // 5. Write Subject Lists

            string subjectdir = Path.Combine(outputdir, "Grouped by Subjects");
            Directory.CreateDirectory(subjectdir);

            foreach (Subject subject in Subjects)
                File.WriteAllText(Path.Combine(subjectdir, subject.Name + ".txt"), subject.StudentListToString());



            // 6. Write School Lists

            if (SchoolDataEnabled)
            {

                string schooldir = Path.Combine(outputdir, "Grouped by School");
                Directory.CreateDirectory(schooldir);

                string schoolhighscoresrawdir = Path.Combine(schooldir, "Sorted by Highest Scores (Raw)");
                Directory.CreateDirectory(schoolhighscoresrawdir);

                string schooltotalscoresrawdir = Path.Combine(schooldir, "Sorted by Total Scores (Raw)");
                Directory.CreateDirectory(schooltotalscoresrawdir);

                string schoolhighscoresscaleddir = "", schooltotalscoresscaleddir = "";

                if (ScalingEnabled)
                {

                    schoolhighscoresscaleddir = Path.Combine(schooldir, "Sorted by Highest Scores (Scaled)");
                    Directory.CreateDirectory(schoolhighscoresscaleddir);

                    schooltotalscoresscaleddir = Path.Combine(schooldir, "Sorted by Total Scores (Scaled)");
                    Directory.CreateDirectory(schooltotalscoresscaleddir);

                }


                foreach (School school in Schools.Values)
                {

                    Student[] schoolstudents = school.Students.Values.ToArray();

                    string headerline = String.Format("{0} ({1} total)" + Environment.NewLine + Environment.NewLine, school.Name, school.ScoreCounts.Sum());

                    string schoolstudenthighscorespath = Path.Combine(schoolhighscoresrawdir, school.SafeName + ".txt");
                    File.WriteAllText(schoolstudenthighscorespath, headerline + StudentsToString(schoolstudents, SortingMethods.HighestScores, ScalingMethods.RawScores, SchoolNameOptions.NoName));

                    string schoolstudenttotalscorespath = Path.Combine(schooltotalscoresrawdir, school.SafeName + ".txt");
                    File.WriteAllText(schoolstudenttotalscorespath, headerline + StudentsToString(schoolstudents, SortingMethods.TotalScores, ScalingMethods.RawScores, SchoolNameOptions.NoName));

                    if (ScalingEnabled)
                    {

                        string schoolstudenthighscoresscaledpath = Path.Combine(schoolhighscoresscaleddir, school.SafeName + ".txt");
                        File.WriteAllText(schoolstudenthighscoresscaledpath, headerline + StudentsToString(schoolstudents, SortingMethods.HighestScores, ScalingMethods.ScaledScores, SchoolNameOptions.NoName));

                        string schoolstudenttotalscoresscaledpath = Path.Combine(schooltotalscoresscaleddir, school.SafeName + ".txt");
                        File.WriteAllText(schoolstudenttotalscoresscaledpath, headerline + StudentsToString(schoolstudents, SortingMethods.TotalScores, ScalingMethods.ScaledScores, SchoolNameOptions.NoName));

                    }

                }

            }



            // 7. Write Subjects grouped by School

            if (SchoolDataEnabled)
            {

                string subjectschooldir = Path.Combine(outputdir, "Grouped by Subject and School");
                Directory.CreateDirectory(subjectschooldir);

                string subjectschoolhighscorescountdir = Path.Combine(subjectschooldir, "Score Counts Sorted by Highest Scores");
                Directory.CreateDirectory(subjectschoolhighscorescountdir);

                string subjectschooltotalscorescountdir = Path.Combine(subjectschooldir, "Score Counts Sorted by Total Scores");
                Directory.CreateDirectory(subjectschooltotalscorescountdir);

                string subjectschoolhighscoresdir = Path.Combine(subjectschooldir, "Student Lists Sorted by Highest Scores");
                Directory.CreateDirectory(subjectschoolhighscoresdir);

                string subjectschooltotalscoresdir = Path.Combine(subjectschooldir, "Student Lists Sorted by Total Scores");
                Directory.CreateDirectory(subjectschooltotalscoresdir);

                foreach (KeyValuePair<Subject, Dictionary<string, School>> subjectschoolpair in SubjectSchools)
                {

                    // write score counts

                    School[] subjectschools = subjectschoolpair.Value.Values.ToArray();

                    // write subject name at the top of the list
                    string headerline = String.Format("{0} ({1} total)" + Environment.NewLine + Environment.NewLine, subjectschoolpair.Key.Name, subjectschoolpair.Key.ScoreCounts.Sum());

                    // write also the scaling correspondences if present
                    if (ScalingEnabled)
                    {
                        headerline += GetScalingLine(subjectschoolpair.Key) + Environment.NewLine + Environment.NewLine;
                    }

                    // write lists for each school
                    foreach (School subjectschool in subjectschoolpair.Value.Values)
                    {

                        // score counts only, high ordering
                        string subjectschoolscorecountshighscorespath = Path.Combine(subjectschoolhighscorescountdir, subjectschoolpair.Key.Name + ".txt");
                        File.WriteAllText(subjectschoolscorecountshighscorespath, headerline + SchoolScoreCountsToString(subjectschools, SortingMethods.HighestScores, ScalingMethods.RawScores));

                        // score counts only, total ordering
                        string subjectschoolscorecountstotalscorespath = Path.Combine(subjectschooltotalscorescountdir, subjectschoolpair.Key.Name + ".txt");
                        File.WriteAllText(subjectschoolscorecountstotalscorespath, headerline + SchoolScoreCountsToString(subjectschools, SortingMethods.TotalScores, ScalingMethods.RawScores));

                        // student lists, high ordering
                        string subjectschoolscoreshighscorespath = Path.Combine(subjectschoolhighscoresdir, subjectschoolpair.Key.Name + ".txt");
                        File.WriteAllText(subjectschoolscoreshighscorespath, headerline + SubjectSchoolStudentsToString(subjectschoolpair.Key, subjectschools, SortingMethods.HighestScores));

                        // student lists, total ordering
                        string subjectschoolscorestotalscorespath = Path.Combine(subjectschooltotalscoresdir, subjectschoolpair.Key.Name + ".txt");
                        File.WriteAllText(subjectschoolscorestotalscorespath, headerline + SubjectSchoolStudentsToString(subjectschoolpair.Key, subjectschools, SortingMethods.TotalScores));

                    }

                }

            }
            





        }



        /// <summary>
        /// Returns a string containing various statistics about the data.
        /// </summary>
        /// <returns>String containing statistics about the data.</returns>
        static string StatisticsToString()
        {

            StringBuilder result = new StringBuilder();


            result.AppendLine(String.Format("Total Students: {0}", Students.Count));

            result.AppendLine();

            result.AppendLine(String.Format("Total Scores: {0}", RecordCount));
            result.AppendLine(String.Format("Average Number of {0}+ Scores per Student: {1}", LowestRawScore, Math.Round((double)RecordCount / Students.Count, 2)));

            result.AppendLine();


            if (SchoolDataEnabled)
            {
                result.AppendLine(String.Format("Total Schools: {0}", Schools.Count));
                result.AppendLine(String.Format("Average Number of {0}+ Scores per School: {1}", LowestRawScore, Math.Round((double)RecordCount / Schools.Count, 2)));

                result.AppendLine();
            }

            result.AppendLine(String.Format("Total Subjects: {0}", Subjects.Count));
            result.AppendLine(String.Format("Average Number of {0}+ Scores per Subject: {1}", LowestRawScore, Math.Round((double)RecordCount / Subjects.Count, 2)));

            result.AppendLine();

            if (ScalingEnabled)
            {
                result.AppendLine(String.Format("Total Scores Scaled Up: {0} ({1:0.00}%)", ScoresScaledUp, (double) 100 * ScoresScaledUp / RecordCount));
                result.AppendLine(String.Format("Total Scores Scaled Down: {0} ({1:0.00}%)", ScoresScaledDown, (double) 100 * ScoresScaledDown / RecordCount));
                result.AppendLine(String.Format("Total Scores Scaled Sideways: {0} ({1:0.00}%)", ScoresScaledSideways, (double) 100 * ScoresScaledSideways / RecordCount));
                result.AppendLine(String.Format("Total Scores Not Scaled: {0} ({1:0.00}%)", ScoresNotScaled, (double)100 * ScoresNotScaled / RecordCount));

                result.AppendLine();

                result.AppendLine(String.Format("Total Scores Above {0}: {1}", MaximumScoreStandard, ScoresAboveMax));
                result.AppendLine(String.Format("Average Number of Scores Above {0} per Student: {1}", MaximumScoreStandard, Math.Round((double)ScoresAboveMax / Students.Count, 2)));

                result.AppendLine();
            }

            result.AppendLine("Total Score Counts (Raw):");
            for (int i = 0; i < ScoreLevelCount; i++)
                result.AppendLine(String.Format("\t{0}: {1}", i + LowestRawScore, ScoreCounts[i]));

            if (ScalingEnabled)
            {
                result.AppendLine();

                result.AppendLine("Total Score Counts (Scaled):");

                bool foundscore = false;

                for (int i = MinimumScore; i <= MaximumScaledScoreSpecial; i++)
                {
                    if (!foundscore) foundscore = (ScaledScoreCounts[i] > 0);

                    if (foundscore)
                        result.AppendLine(String.Format("\t{0}: {1}", i, ScaledScoreCounts[i]));
                }

            }


            return result.ToString().TrimEnd();

        }



        /// <summary>
        /// Returns a string containing the raw to scaled score correspondences for each subject.
        /// </summary>
        /// <returns>String containing raw to scaled score correspondence for each subject.</returns>
        static string ScalingDataToString()
        {

            StringBuilder result = new StringBuilder();


            // find longest study name
            int longest = 0;
            foreach (Subject subject in Subjects)
                if (subject.Name.Length > longest) longest = subject.Name.Length;

            // number of spaces between end of longest study name and start of score line
            int offset = 3;

            // write raw score line
            result.Append(new string(' ', longest + offset - 1));
            int scorepartlength = 0;
            for (int i = LowestRawScore; i < LowestRawScore + ScoreLevelCount; i++)
            {
                result.Append(i.ToString().PadLeft(7, ' '));
                scorepartlength += i.ToString().PadLeft(7, ' ').Length;
            }
            scorepartlength--;
            result.AppendLine();

            Subject[] subjects = Subjects.ToArray();

            Array.Sort(subjects);

            // write each subject line
            foreach (Subject subject in subjects)
            {
                result.Append(subject.Name.PadRight(longest + offset - 1, ' '));

                if (subject.ScalingData != null)
                {
                    for (int i = LowestRawScore; i < LowestRawScore + ScoreLevelCount; i++)
                        result.Append(String.Format("  {0:0.00}", Math.Round(subject.ScalingData.GetScaledScore(i), 2)));
                    result.AppendLine();
                }
                else
                {
                    // centres error message
                    string errormessage = "*** NO SCALING DATA ***";
                    errormessage = errormessage.PadLeft(((scorepartlength - errormessage.Length) / 2) + errormessage.Length, ' ');
                    errormessage = "  " + errormessage.PadRight(scorepartlength - 1, ' ');
                    result.AppendLine(errormessage);
                }
            }

            return result.ToString().TrimEnd();

        }



        /// <summary>
        /// Returns a string containing a ranked list of students and their scores.
        /// </summary>
        /// <param name="students">Array of students to create list from.</param>
        /// <param name="sortingmethod">Specifies the sorting method (e.g. by sum of scores).</param>
        /// <param name="scalingmethod">Specifies whether to use raw or scaled scores in ranking.</param>
        /// <returns>String containing ranked list of students and their scores.</returns>
        static string StudentsToString(Student[] students, SortingMethods sortingmethod, ScalingMethods scalingmethod, SchoolNameOptions schoolnameoption)
        {

            StringBuilder result = new StringBuilder();


            // sort according to arguments

            if (sortingmethod == SortingMethods.HighestScores && scalingmethod == ScalingMethods.RawScores)
                Array.Sort(students, Sorting.StudentHighScoreCompareRaw);

            else if (sortingmethod == SortingMethods.HighestScores && scalingmethod == ScalingMethods.ScaledScores)
                Array.Sort(students, Sorting.StudentHighScoreCompareScaled);

            else if (sortingmethod == SortingMethods.TotalScores && scalingmethod == ScalingMethods.RawScores)
                Array.Sort(students, Sorting.StudentTotalScoreCompareRaw);

            else if (sortingmethod == SortingMethods.TotalScores && scalingmethod == ScalingMethods.ScaledScores)
                Array.Sort(students, Sorting.StudentTotalScoreCompareScaled);


            // write each line
            for (int i = 0; i < students.Length; i++)
            {
                result.AppendLine(String.Format("{0}. {1}", i + 1, students[i].ToStringFull(schoolnameoption)));
                result.AppendLine();
            }

            
            return result.ToString().TrimEnd();

        }



        /// <summary>
        /// Produces a sorted string containing the number of times each school in the provided array achieved each raw or scaled score.
        /// </summary>
        /// <param name="schools">The list of schools to include.</param>
        /// <param name="sortingmethod">The method in which the list should be sorted.</param>
        /// <param name="scalingmethod">Determines whether to use raw or scaled scores (if present) when sorting.</param>
        /// <returns>String containing list of schools and the number of times each received each raw or scaled score.</returns>
        static string SchoolScoreCountsToString(School[] schools, SortingMethods sortingmethod, ScalingMethods scalingmethod)
        {

            StringBuilder result = new StringBuilder();


            // sort according to arguments

            if (scalingmethod == ScalingMethods.RawScores)
            {

                if (sortingmethod == SortingMethods.HighestScores && scalingmethod == ScalingMethods.RawScores)
                    Array.Sort(schools, Sorting.SchoolScoreCountHighCompareRaw);

                else if (sortingmethod == SortingMethods.TotalScores && scalingmethod == ScalingMethods.RawScores)
                    Array.Sort(schools, Sorting.SchoolScoreCountTotalCompareRaw);

                // write each line
                for (int i = 0; i < schools.Length; i++)
                    result.AppendLine(String.Format("{0}. {1}", i + 1, schools[i].RawScoreCountsToString()));

            }
            else if (scalingmethod == ScalingMethods.ScaledScores)
            {
                if (sortingmethod == SortingMethods.HighestScores && scalingmethod == ScalingMethods.ScaledScores)
                    Array.Sort(schools, Sorting.SchoolScoreCountHighCompareScaled);

                else if (sortingmethod == SortingMethods.TotalScores && scalingmethod == ScalingMethods.ScaledScores)
                    Array.Sort(schools, Sorting.SchoolScoreCountTotalCompareScaled);

                // write each line
                for (int i = 0; i < schools.Length; i++)
                    result.AppendLine(String.Format("{0}. {1}", i + 1, schools[i].ScaledScoreCountsToString()));
            }


            return result.ToString().TrimEnd();

        }



        /// <summary>
        /// Returns a string with a list of schools and their students for a specific subject.
        /// </summary>
        /// <param name="subject">The subject to make the list for.</param>
        /// <param name="subjectschools">An array containing the schools who had students take the subject.</param>
        /// <param name="sortingmethod">Specifies the sorting method.</param>
        /// <returns>String containing list of schools and students for the subject.</returns>
        static string SubjectSchoolStudentsToString(Subject subject, School[] subjectschools, SortingMethods sortingmethod)
        {

            StringBuilder result = new StringBuilder();


            // sort according to arguments

            if (sortingmethod == SortingMethods.HighestScores)
                Array.Sort(subjectschools, Sorting.SchoolScoreCountHighCompareRaw);

            else if (sortingmethod == SortingMethods.TotalScores)
                Array.Sort(subjectschools, Sorting.SchoolScoreCountTotalCompareRaw);

            // write each line
            for (int i = 0; i < subjectschools.Length; i++)
            {
                // write the school name and ranking
                result.AppendLine(String.Format("{0}. {1} ({2} total)", i + 1, subjectschools[i].Name, subjectschools[i].ScoreCounts.Sum()));


                // list students at each score level
                List<Student>[] subjectstudents = new List<Student>[ScoreLevelCount];
                for (int j = 0; j < subjectstudents.Length; j++)
                    subjectstudents[j] = new List<Student>(100);

                foreach (Student student in subjectschools[i].Students.Values)
                    subjectstudents[(int)student.GetScore(subject) - LowestRawScore].Add(student);


                // write the list of students at each score
                for (int j = LowestRawScore + ScoreLevelCount - 1; j >= LowestRawScore; j--)
                {
                    // only list students if there are students to list!
                    if (subjectschools[i].GetScoreCount(j) > 0)
                    {

                        result.AppendLine(String.Format("\t{0} ({1} total):", j, subjectschools[i].GetScoreCount(j)));

                        // write each student name
                        foreach (Student student in subjectstudents[j - LowestRawScore])
                            result.AppendLine("\t\t" + student.Name);

                    }
                }

                result.AppendLine();
            }


            return result.ToString().TrimEnd();

        }



        /// <summary>
        /// Returns a string containing the raw scores and corresponding for the subject provided.
        /// </summary>
        /// <param name="subject">Subject for which scaling data should be listed.</param>
        /// <returns>String containing raw and scaled score correspondences.</returns>
        public static string GetScalingLine(Subject subject)
        {
            if (subject.HasScalingData)
            {
                StringBuilder scalingline = new StringBuilder();

                scalingline.AppendLine("Scaling:");

                // raw scores
                for (int i = LowestRawScore + ScoreLevelCount - 1; i >= LowestRawScore; i--)
                    scalingline.Append(i.ToString().PadLeft((i == LowestRawScore + ScoreLevelCount - 1 ? 5 : 7), ' '));
                scalingline.AppendLine();

                // scaled scores
                if (subject.ScalingData != null)
                    for (int i = LowestRawScore + ScoreLevelCount - 1; i >= LowestRawScore; i--)
                        scalingline.Append(String.Format("{0}{1:0.00}", (i == LowestRawScore + ScoreLevelCount - 1 ? "" : "  "), Math.Round(subject.ScalingData.GetScaledScore(i), 2)));

                return scalingline.ToString();
            }
            else
            {
                return "No scaling data (small study)";
            }
        }

    }
}
