﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace StudyScores
{
    /// <summary>
    /// Represents a student. Contains a reference to the student's school and a list of subjects and scores.
    /// </summary>
    class Student : IComparable
    {

        /// <summary>
        /// The student's name.
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// List of scores, corresponding to <see cref="Scores" />
        /// </summary>
        public List<int> Scores { get; set; }

        /// <summary>
        /// List of subjects that the student took.
        /// </summary>
        public List<Subject> Subjects { get; set; }

        /// <summary>
        /// The student's school.
        /// </summary>
        public School School { get; set; }

        /// <summary>
        /// Sum of the student's raw scores.
        /// </summary>
        public int ScoreSum {
            get {
                return Scores.Sum();
            }
        }

        /// <summary>
        /// Sum of the student's scaled scores.
        /// </summary>
        public double ScaledScoreSum
        {
            get
            {
                double sum = 0;
                for (int i = 0; i < Scores.Count; i++)
                    sum += Subjects[i].GetScaledScore(Scores[i]);
                return sum;
            }
        }

        public Student()
        {
            Scores = new List<int>(10);
            Subjects = new List<Subject>(10);
        }

        public override string ToString()
        {
            return Name;
        }

        public int CompareTo(object obj)
        {
            Student student = obj as Student;
            if (student != null)
                return Name.CompareTo(student.Name);
            else
                throw new ArgumentException("Must compare to another Student.");
        }


        /// <summary>
        /// Returns a string with the student's name, school and scores.
        /// </summary>
        /// <returns>String containing student name, school name and list of scores.</returns>
        public string ToStringFull(Program.SchoolNameOptions schoolnameoption)
        {
            StringBuilder result = new StringBuilder();

            // write name and school
            result.Append(Name);
            if (Program.SchoolDataEnabled && schoolnameoption == Program.SchoolNameOptions.IncludeName)
                result.AppendFormat(" ({0})", School.Name);
            result.AppendLine();

            result.AppendFormat("\t{0} total (Raw Sum: {1}", Subjects.Count, ScoreSum);

            if (Program.ScalingEnabled)
                result.AppendFormat(", Scaled Sum: {0}", ScaledScoreSum);

            result.AppendLine(")");


            // write subject scores
            for (int i = 0; i < Scores.Count; i++)
            {
                // write subject name and raw score
                result.Append(String.Format("\t\t{0}: {1}", Subjects[i].Name, Scores[i]));

                // write scaled score if present
                if (Subjects[i].HasScalingData)
                    result.Append(String.Format(" ({0})", Subjects[i].GetScaledScore(Scores[i])));

                result.AppendLine();
            }

            return result.ToString().TrimEnd();
        }



        /// <summary>
        /// Adds the specified student (if not already present) and score.
        /// </summary>
        /// <param name="student">Student to be added (or ignored if already present).</param>
        /// <param name="score">Score to be added.</param>
        public void AddScore(Subject subject, int score)
        {
            if (score < Program.LowestRawScore || score > Program.LowestRawScore + Program.ScoreLevelCount - 1)
                throw new ArgumentOutOfRangeException(String.Format("Score must be between {0} and {1}. Input: {2}.", Program.LowestRawScore, Program.LowestRawScore + Program.ScoreLevelCount - 1, score));
            Subjects.Add(subject);
            Scores.Add(score);
        }



        /// <summary>
        /// Returns the score corresponding to the specified subject, if present. Returns null if not found.
        /// </summary>
        /// <param name="subject">The subject whose score should be returned.</param>
        /// <returns>The score corresponding to the specified subject, or null if the specified subject could not be found.</returns>
        public int? GetScore(Subject subject)
        {
            int scoreindex = Subjects.IndexOf(subject);
            return (scoreindex != -1 ? (int?)Scores[scoreindex] : null);
        }



        /// <summary>
        /// Returns an array specifying the number of times the student received each raw score.
        /// </summary>
        /// <returns>Array containing the number of times each raw score was achieved.</returns>
        public int[] GetRawScoreCounts()
        {
            int[] counts = new int[Program.ScoreLevelCount];

            foreach (int score in Scores)
                counts[score - Program.LowestRawScore]++;

            return counts;
        }

        /// <summary>
        /// Returns an array containing a list of scaled scores attained by the student in descending order.
        /// </summary>
        /// <returns>List of scaled scores.</returns>
        public double[] GetScaledScoresInDescendingOrder()
        {
            int[] scores = Scores.ToArray();
            double[] scaledscores = new double[scores.Length];
            
            for (int i = 0; i < scores.Length; i++)
                scaledscores[i] = Subjects[i].GetScaledScore(scores[i]);
            
            Array.Sort(scaledscores);
            Array.Reverse(scaledscores);

            return scaledscores;
        }
    }
}
